import os
import argparse
import json
from dotenv import load_dotenv
from datetime import datetime
from pathlib import Path
from webdav3.client import Client
import pyzipper
import sys


def decrypt_zip(zip_file_path, password):
    """
    Decrypts a password-protected zip file that contains exactly one file.

    Args:
        zip_file_path (str): Path to the zip file.
        password (str): Password for decrypting the zip file.

    Returns:
        str: Name of the extracted file.

    Raises:
        SystemExit: If the zip file does not exist, contains multiple files, or decryption fails.
    """
    if not os.path.exists(zip_file_path):
        print(f"The file {zip_file_path} does not exist.")
        sys.exit(1)

    try:
        with pyzipper.AESZipFile(zip_file_path) as zf:
            # Set the password for decryption
            zf.setpassword(password.encode("utf-8"))
            files_info = zf.infolist()

            # Ensure there is only one file in the zip
            if len(files_info) != 1:
                print("The zip file must contain exactly one file.")
                sys.exit(1)

            # Extract and return the name of the file
            file_name = files_info[0].filename
            zf.extractall(pwd=password.encode("utf-8"))
            return file_name

    except (RuntimeError, Exception) as e:
        print(f"Decryption error: {e}")
        sys.exit(1)


def load_json(file):
    """
    Loads a JSON file and returns its contents.

    Args:
        file (str): Path to the JSON file.

    Returns:
        dict: Parsed contents of the JSON file.

    Raises:
        SystemExit: If the file cannot be found or has an invalid JSON format.
    """
    try:
        with open(file, "r") as file:
            return json.load(file)
    except json.JSONDecodeError:
        print("Invalid JSON format.")
        sys.exit(1)
    except FileNotFoundError:
        print(f"File {file} not found.")
        sys.exit(1)


def get_workflow_name():
    """
    Retrieves or generates a workflow name based on a timestamp.

    Returns:
        str: Workflow name from the existing file or a new timestamped name.
    """
    rce_workflow_info = Path("rce-workflow-info.txt")
    if rce_workflow_info.is_file():
        with rce_workflow_info.open("r") as file:
            return file.read().split("Workflow name: ")[-1].strip()
    # Generate a new name based on the current timestamp if no file exists
    return datetime.now().strftime("results_%Y-%m-%d_%H:%M:%S_%f")[:26]


def upload(client, local_path, remote_path):
    """
    Uploads all files from a local directory to a remote directory on a WebDAV server.

    Args:
        client (Client): WebDAV client instance.
        local_path (str): Path to the local directory.
        remote_path (Path): Path to the remote directory.
    """
    for item in Path(local_path).iterdir():
        remote_path_i = (remote_path / item.name).as_posix()
        print(f'Sync local_path="{item}" with remote_path="{remote_path_i}"')

        # Check if the item is a directory or file, and upload accordingly
        if item.is_dir():
            client.upload_directory(remote_path=remote_path_i, local_path=str(item))
        elif item.is_file():
            client.upload_file(remote_path=remote_path_i, local_path=str(item))
        else:
            print(f"Cannot upload {item}")


def download(client, local_path, remote_path):
    """
    Downloads files or directories from a remote WebDAV directory to a local path.

    Args:
        client (Client): WebDAV client instance.
        local_path (str): Local path where files should be downloaded.
        remote_path (str): Path to the remote directory to download.
    """
    print(f'Sync remote_path="{remote_path}" with local_path="{local_path}"')
    client.download_sync(remote_path, local_path)


def main(credentials_file, config_file):
    """
    Main function to handle the workflow:
        - Loads environment variables
        - Decrypts and loads credentials
        - Sets up WebDAV client and workflow name
        - Uploads specified local directory to the remote server
        - Downloads specified path from remote server to local directory

    Args:
        credentials_file (str): Path to the credentials file (zip or JSON).
        config_file (str): Path to the configuration file (JSON).
    """
    load_dotenv()

    # Determine file type for the credentials file
    file_type = Path(credentials_file).suffix

    # Decrypt if it's a zip file, then load credentials
    if file_type == ".zip":
        credentials_file = decrypt_zip(
            credentials_file, os.getenv("CREDENTIALS_ENCRYPTION_PWD")
        )
        file_type = Path(credentials_file).suffix

    # Ensure credentials file is a JSON
    if file_type != ".json":
        print(f"File suffix {file_type} not supported. Please use JSON.")
        sys.exit(1)

    # Load credentials and instantiate WebDAV client
    credentials = load_json(credentials_file)
    os.remove(credentials_file)
    client = Client(credentials)

    # Load configuration
    config = load_json(config_file)
    upload_config = config.get("upload")
    download_config = config.get("download")

    # Perform uploads
    if upload_config.get("perform_upload", False):

        # Generate or retrieve workflow name
        workflow_name = get_workflow_name()
        print(f"Workflow name: {workflow_name}")

        # Load configuration and set up remote directory path

        remote_path = Path(upload_config.get("remote_path")) / workflow_name
        client.mkdir(str(remote_path.as_posix()))

        # Upload specified directory to the remote server
        local_path = Path(upload_config.get("local_path"))

        upload(client, local_path, remote_path)

    if download_config.get("perform_download", False):
        remote_path = Path(download_config.get("remote_path")).as_posix()
        local_path = download_config.get("local_path")
        download(client, local_path, remote_path)


# Perform downloads
def parse_args():
    """
    Parses command-line arguments.

    Returns:
        tuple: (credentials_file, config_file, upload_directory, download_directory)
    """
    parser = argparse.ArgumentParser(
        prog="NextCloud Interface",
        description="This tool provides synchronization with NextCloud for RCE workflows",
    )
    parser.add_argument(
        "--credentials_file",
        default="credentials.zip",
        help='Path of the NextCloud credentials file (json or zip). Default is "credentials.zip"',
    )
    parser.add_argument(
        "--config_file",
        default="config.json",
        help='Path of the configuration file (json). Default is "config.json"',
    )
    args = parser.parse_args()

    return (args.credentials_file, args.config_file)


if __name__ == "__main__":
    # Parse arguments and execute main workflow
    credentials_file, config_file = parse_args()
    main(credentials_file, config_file)
